<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\Hash;
use App\Http\Controllers\Controller;
use App\Models\user;
use Illuminate\Http\Request;
use Spatie\Permission\Traits\HasRoles;
use Exception;

class UsersController extends Controller
{

      use HasRoles;
    // /**
    //  * Display a listing of the users.
    //  *
    //  * @return \Illuminate\View\View
    //  */
   public function index()
    {
        $users = User::with('roles')->get()->map(function ($user) {
            return [
                'id'    => $user->id,
                'name'  => $user->name,
                'email' => $user->email,
                'role'  => optional($user->roles->first())->name,
            ];
        });

        return response()->json($users);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name'     => 'required|string|max:255',
            'email'    => 'nullable|string|email|unique:users',
            'password' => 'nullable|string',
            'role'     => 'nullable|string|exists:roles,name',
            'category' => 'nullable|string',
        ]);

        $user = User::create([
            'name'     => $request->name,
            'email'    => $request->email,
            'password' => Hash::make($request->password ?? '12345678'),
            'category' => $request->category,
        ]);

        if ($request->filled('role')) {
            $user->assignRole($request->role);
        }

        return response()->json([
            'id'    => $user->id,
            'name'  => $user->name,
            'email' => $user->email,
            'role'  => optional($user->roles->first())->name,
        ]);
    }

    public function show($id)
    {
        $user = User::with('roles')->findOrFail($id);

        return response()->json([
            'id'    => $user->id,
            'name'  => $user->name,
            'email' => $user->email,
            'role'  => optional($user->roles->first())->name,
        ]);
    }

    public function update($id, Request $request)
    {
        $request->validate([
            'name'     => 'required|string',
            'email'    => 'nullable|string|email|unique:users,email,' . $id,
            'password' => 'nullable|string',
            'role'     => 'nullable|string|exists:roles,name',
            'category' => 'nullable|string',
        ]);

        $user = User::findOrFail($id);

        $user->name     = $request->name;
        $user->email    = $request->email;
        $user->category = $request->category;

        if ($request->filled('password')) {
            $user->password = Hash::make($request->password);
        }

        $user->save();

        if ($request->filled('role')) {
            $user->syncRoles([$request->role]);
        }

        return response()->json([
            'id'    => $user->id,
            'name'  => $user->name,
            'email' => $user->email,
            'role'  => optional($user->roles->first())->name,
        ]);
    }

    public function destroy($id)
    {
        try {
            $user = User::findOrFail($id);
            $user->delete();

            return response()->json(['message' => 'User deleted successfully']);
        } catch (Exception $e) {
            return response()->json(['message' => 'User not deleted successfully'], 500);
        }
    }
}
