<?php

namespace App\Http\Controllers;

use App\Models\Item;
use App\Models\ItemType;
use App\Models\Sell;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class SallesDashboardController extends Controller
{
    public function getDashboardData(Request $request)
    {
        $timeRange = $request->input('time_range', 'month'); // day, week, month, year
        
        try {
            return response()->json([
                'stats' => $this->getStatsCards($timeRange),
                'salesData' => $this->getSalesData($timeRange),
                'topProducts' => $this->getTopProducts(),
                'categories' => $this->getCategoriesData()
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to fetch dashboard data',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    private function getStatsCards($timeRange)
    {
        $startDate = $this->getStartDate($timeRange);
        $previousStartDate = $this->getPreviousStartDate($timeRange);
        
        $revenueChange = $this->calculateRevenueChange($timeRange);
        $ordersChange = $this->calculateOrdersChange($timeRange);
        
        return [
            [
                'title' => 'Total Revenue',
                'value' => '$' . number_format($this->calculateTotalRevenue($startDate)),
                'change' => round($revenueChange, 2) . '%',
                'trend' => $this->calculateTrend($revenueChange),
                'icon' => 'AttachMoney',
                'color' => '#4caf50'
            ],
            [
                'title' => 'Orders',
                'value' => number_format($this->calculateTotalOrders($startDate)),
                'change' => round($ordersChange, 2) . '%',
                'trend' => $this->calculateTrend($ordersChange),
                'icon' => 'ShoppingCart',
                'color' => '#2196f3'
            ],
            [
                'title' => 'Inventory Items',
                'value' => number_format($this->calculateTotalInventory()),
                'change' => $this->calculateInventoryChange($timeRange) . '%',
                'trend' => $this->calculateTrend($this->calculateInventoryChange($timeRange)),
                'icon' => 'Inventory',
                'color' => '#ff9800'
            ],
            [
                'title' => 'Active Products',
                'value' => number_format($this->calculateActiveProducts()),
                'change' => $this->calculateProductsChange($timeRange) . '%',
                'trend' => $this->calculateTrend($this->calculateProductsChange($timeRange)),
                'icon' => 'Category',
                'color' => '#f44336'
            ]
        ];
    }

    private function getSalesData($timeRange)
    {
        $salesData = [];
        $periods = $this->getPeriods($timeRange);
        
        foreach ($periods as $period) {
            $salesData[] = [
                'date' => $period['label'],
                'sales' => $this->getSalesForPeriod($period['start'], $period['end']),
                'orders' => $this->getOrdersForPeriod($period['start'], $period['end']),
                'revenue' => $this->getRevenueForPeriod($period['start'], $period['end'])
            ];
        }
        
        return $salesData;
    }

    private function getTopProducts()
    {
     $sales = Sell::with('stock.type') // بارگذاری رابطه با آیتم
    // ->select(
    //     'sells.id',
    //     'sells.purchase_price as price',
    //     'sells.qty as sold',
    //     DB::raw('(sells.qty * sells.sell_price) as revenue'),
    //     'sells.dateInsert'
    // )
    ->where('sells.isdelete', 0)
    ->orderBy('sells.dateInsert', 'DESC')
    ->get();

return $sales->map(function ($item) {
    return [
        'id' => $item->id,
        'name' => $item->stock->name ?? 'N/A', // دسترسی از طریق رابطه
        'category' => $item->stock->type_id ? $this->getCategoryName($item->stock->type_id) : 'N/A',
        'price' => number_format($item->purchase_price, 2),
        'sold' => (int) $item->qty,
        'revenue' => number_format($item->qty*$item->sell_price, 2),
        'date' => $item->dateInsert,
    ];
})->toArray();
    }

    private function getCategoriesData()
    {
        // Check if types table exists
        // if (!DB::getSchemaBuilder()->hasTable('types')) {
        //     // Fallback: get categories from items if types table doesn't exist
        //     $categories = DB::table('items')
        //         ->select(
        //             DB::raw('type_id as id'),
        //             DB::raw('"name" as name'),
        //             DB::raw('COUNT(*) * 10 as value') // Simple fallback value
        //         )
        //         ->where('isdelete', 0)
        //         ->groupBy('type_id')
        //         ->get();
        // } else {
        //     $categories = DB::table('sells')
        //         ->join('items', 'sells.item_id', '=', 'items.id')
        //         ->join('types', 'items.type_id', '=', 'types.id')
        //         ->select(
        //             'types.id',
        //             'types.name',
        //             DB::raw('ROUND((COUNT(sells.id) / (SELECT COUNT(*) FROM sells WHERE isdelete = 0)) * 100) as value')
        //         )
        //         ->where('sells.isdelete', 0)
        //         ->where('items.isdelete', 0)
        //         ->groupBy('types.id', 'types.name')
        //         ->orderBy('value', 'DESC')
        //         ->get();
        // }

        // return $categories->map(function ($category) {
        //     return [
        //         'id' => $category->id,
        //         'name' => $category->name,
        //         'value' => (int)$category->value
        //     ];
        // })->toArray();
        return ItemType::select('id', 'name')->get()->toArray();

    }

    // Helper methods
    private function calculateTotalRevenue($startDate = null)
    {
        $query = DB::table('sells')
            ->where('isdelete', 0);
            
        if ($startDate) {
            $query->where('dateInsert', '>=', $startDate);
        }
        
        $result = $query->sum(DB::raw('qty * sell_price'));
        return $result ? $result : 0;
    }

    private function calculateTotalOrders($startDate = null)
    {
        $query = DB::table('sells')
            ->select(DB::raw('COUNT(DISTINCT bill_id) as orders'))
            ->where('isdelete', 0);
            
        if ($startDate) {
            $query->where('dateInsert', '>=', $startDate);
        }
        
        $result = $query->first();
        return $result ? $result->orders : 0;
    }

    private function calculateTotalInventory()
    {
        if (!DB::getSchemaBuilder()->hasTable('purchases')) {
            return DB::table('items')
                ->where('isdelete', 0)
                ->sum('qty');
        }
        
        return DB::table('purchases')
            ->where('isdelete', 0)
            ->sum('remainqty');
    }

    private function calculateActiveProducts()
    {
        return DB::table('items')
            ->where('isdelete', 0)
            ->where('qty', '>', 0)
            ->count();
    }

    private function calculateRevenueChange($timeRange)
    {
        $current = $this->calculateTotalRevenue($this->getStartDate($timeRange));
        $previous = $this->calculateTotalRevenue($this->getPreviousStartDate($timeRange));
        
        if ($previous == 0) return $current > 0 ? 100 : 0;
        return (($current - $previous) / $previous) * 100;
    }

    private function calculateOrdersChange($timeRange)
    {
        $current = $this->calculateTotalOrders($this->getStartDate($timeRange));
        $previous = $this->calculateTotalOrders($this->getPreviousStartDate($timeRange));
        
        if ($previous == 0) return $current > 0 ? 100 : 0;
        return (($current - $previous) / $previous) * 100;
    }

    private function calculateInventoryChange($timeRange)
    {
        // Simple implementation - you might want to track inventory changes over time
        return rand(-5, 15);
    }

    private function calculateProductsChange($timeRange)
    {
        // Simple implementation
        return rand(5, 25);
    }

    private function calculateTrend($change)
    {
        return $change >= 0 ? 'up' : 'down';
    }

    private function getStartDate($timeRange)
    {
        return match($timeRange) {
            'day' => Carbon::now()->subDay(),
            'week' => Carbon::now()->subWeek(),
            'month' => Carbon::now()->subMonth(),
            'year' => Carbon::now()->subYear(),
            default => Carbon::now()->subMonth()
        };
    }

    private function getPreviousStartDate($timeRange)
    {
        return match($timeRange) {
            'day' => Carbon::now()->subDays(2),
            'week' => Carbon::now()->subWeeks(2),
            'month' => Carbon::now()->subMonths(2),
            'year' => Carbon::now()->subYears(2),
            default => Carbon::now()->subMonths(2)
        };
    }

    private function getPeriods($timeRange)
    {
        $periods = [];
        $now = Carbon::now();
        
        switch ($timeRange) {
            case 'day':
                for ($i = 0; $i < 24; $i++) {
                    $periods[] = [
                        'start' => $now->copy()->subHours(23 - $i)->startOfHour(),
                        'end' => $now->copy()->subHours(23 - $i)->endOfHour(),
                        'label' => $now->copy()->subHours(23 - $i)->format('H:00')
                    ];
                }
                break;
                
            case 'week':
                for ($i = 0; $i < 7; $i++) {
                    $periods[] = [
                        'start' => $now->copy()->subDays(6 - $i)->startOfDay(),
                        'end' => $now->copy()->subDays(6 - $i)->endOfDay(),
                        'label' => $now->copy()->subDays(6 - $i)->format('D')
                    ];
                }
                break;
                
            case 'month':
                $daysInMonth = $now->daysInMonth;
                for ($i = 0; $i < min(7, $daysInMonth); $i++) {
                    $daysPerPeriod = ceil($daysInMonth / 7);
                    $startDay = $i * $daysPerPeriod + 1;
                    $endDay = min(($i + 1) * $daysPerPeriod, $daysInMonth);
                    
                    $periods[] = [
                        'start' => $now->copy()->startOfMonth()->addDays($startDay - 1),
                        'end' => $now->copy()->startOfMonth()->addDays($endDay - 1)->endOfDay(),
                        'label' => 'Week ' . ($i + 1)
                    ];
                }
                break;
                
            case 'year':
                for ($i = 0; $i < 12; $i++) {
                    $periods[] = [
                        'start' => $now->copy()->subMonths(11 - $i)->startOfMonth(),
                        'end' => $now->copy()->subMonths(11 - $i)->endOfMonth(),
                        'label' => $now->copy()->subMonths(11 - $i)->format('M')
                    ];
                }
                break;
                
            default:
                for ($i = 0; $i < 7; $i++) {
                    $periods[] = [
                        'start' => $now->copy()->subDays(6 - $i)->startOfDay(),
                        'end' => $now->copy()->subDays(6 - $i)->endOfDay(),
                        'label' => $now->copy()->subDays(6 - $i)->format('D')
                    ];
                }
        }
        
        return $periods;
    }

    private function getSalesForPeriod($start, $end)
    {
        return DB::table('sells')
            ->where('isdelete', 0)
            ->whereBetween('dateInsert', [$start, $end])
            ->sum('qty') ?: 0;
    }

    private function getOrdersForPeriod($start, $end)
    {
        return DB::table('sells')
            ->where('isdelete', 0)
            ->whereBetween('dateInsert', [$start, $end])
            ->distinct()
            ->count('bill_id') ?: 0;
    }

    private function getRevenueForPeriod($start, $end)
    {
        $result = DB::table('sells')
            ->where('isdelete', 0)
            ->whereBetween('dateInsert', [$start, $end])
            ->sum(DB::raw('qty * sell_price'));
            
        return $result ? $result / 100 : 0;
    }

    private function getCategoryName($typeId)
    {
        if (!DB::getSchemaBuilder()->hasTable('item_types')) {
            return 'Category ' . $typeId;
        }
        
        $category = DB::table('item_types')->where('id', $typeId)->first();
        return $category ? $category->name : 'Unknown';
    }
}