<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Http\Resources\BelanceResource;
use App\Http\Resources\MoneyResource;
use App\Models\Accounts;
use App\Models\Belances;
use App\Models\Bill;
use App\Models\CabinStock;
use App\Models\Item;
use App\Models\Purchasebills;
use App\Models\Sidebar;
use App\Models\User;
use App\Models\Purchase;
use App\Models\Report;
use App\Models\Moneys;
use App\Models\Stock;
use Illuminate\Http\Request;
use Exception;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class PurchasesController extends Controller
{
        public function dateChange($date)
        {
            if ($date) {
                try {
                    return Carbon::parse($date)->format('Y-m-d H:i:s');
                } catch (\Exception $e) {
                    return response()->json(['error' => 'Invalid date format'], 422);
                }
            }
            return null;
        }

    /**
     * Display a listing of the purchases.
     */
    public function index(Request $request)
    {
        try {
            $isDelete = (int) $request->query('delete', 0);
            $perPage = (int) $request->query('perPage', 10);
            $search = $request->query('search', false);
            $startDate = $request->input('StartDate', false);
            $endDate = $request->input('EndDate', false);
            $user_id = $request->input('user_id', false);
            $currentPage = (int) $request->query('page', 1);

            $query = Purchase::with(['bill.money', 'user', 'bill.accounts.account', 'stock', 'cobin'])
                ->whereHas('bill', function ($query) use ($isDelete) {
                    $query->where('isdelete', $isDelete);
                });

            if ($user_id && $user_id != '0') {
                $query->where('user_id', $user_id);
            }

            if ($startDate && $endDate) {
                $query->whereBetween('dateInsert', [$startDate, $endDate]);
            } elseif ($startDate) {
                $query->where('dateInsert', '>=', $startDate);
            } elseif ($endDate) {
                $query->where('dateInsert', '<=', $endDate);
            }

            if ($search !== false && $search !== 'false' && !empty($search)) {
                $query->whereHas('bill', function ($q) use ($search) {
                    $q->where(function ($inner) use ($search) {
                        $inner->where('temp_customer', 'like', "%$search%")
                              ->orWhereHas('accounts.account', function ($b) use ($search) {
                                  $b->where('name', 'like', "%$search%");
                              });
                    });
                });
            }

            $query->orderBy('dateInsert', 'desc');

            $reports = $query->get()
                ->groupBy(fn($item) => $item->bill->id)
                ->map(function ($group) {
                    $totalInsell = $group->sum(fn($purchase) => $purchase->insell ?? 0);
                    return [
                        'bill' => $group->first()->bill,
                        'money' => $group->first()->bill->money,
                        'purchase' => $group->values()->toArray(),
                        'insell' => $totalInsell,
                    ];
                })
                ->values();

            $paginatedData = array_values(
                $reports->slice(($currentPage - 1) * $perPage, $perPage)->toArray()
            );

            return response()->json([
                'data' => $paginatedData,
                'total' => $reports->count(),
                'current_page' => $currentPage,
                'per_page' => $perPage,
                'last_page' => ceil($reports->count() / $perPage),
            ], 200);

        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to fetch purchases: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a new purchase in the storage.
     */
    public function store(Request $request)
    {
        DB::beginTransaction();
        
        try {
            $existing = $request->query('Exesting');
            $money = $request->query('money');
            $moneyMain = $request->query('money');
            $accounts_id = $request->query('accounts_id');
            $paidAmount = $request->query('PaidAmount', 0);
            $totalAmount = $request->query('TotalAmount', 0);
            $customerName = $request->query('CustomerName');
            $dateInsert = $request->query('DateInsert');
            $purchase_money = $money;

            $validatedData = $request->validate([
                'arr' => 'required|array',
                'arr.*.stocks_id' => 'required',
                'arr.*.accounts_id' => 'nullable',
                'arr.*.qty' => 'required|numeric|min:0',
                'arr.*.weight' => 'nullable',
                'arr.*.dateInsert' => 'required',
                'arr.*.rate' => 'nullable|numeric|min:0',
                'arr.*.user_id' => 'required',
                'arr.*.isdelete' => 'nullable|boolean',
                'arr.*.purchase_price' => 'required|numeric|min:0',
                'arr.*.sell_price' => 'nullable|numeric|min:0',
                'arr.*.expiry_date' => 'nullable',
                'arr.*.description' => 'nullable',
                'arr.*.tepm_customer' => 'nullable',
                'arr.*.money_id' => 'nullable',
                'arr.*.cobin_id' => 'nullable',
                'arr.*.remainqty' => 'nullable',
            ]);

            $dateInsert = $this->parseDate($dateInsert ?? $validatedData['arr'][0]['dateInsert']);

            if ($existing === 'ok') {
                return $this->handleExistingPurchase($validatedData, $money, $accounts_id, $paidAmount, $totalAmount, $dateInsert, $purchase_money);
            } else {
                return $this->handleNewPurchase($validatedData, $money, $totalAmount, $customerName, $dateInsert, $purchase_money);
            }

        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json([
                'error' => 'Validation failed',
                'details' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'error' => 'Failed to create purchase: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Handle existing account purchase
     */
    private function handleExistingPurchase($validatedData, $money, $accounts_id, $paidAmount, $totalAmount, $dateInsert, $purchase_money)
    {
        $balance = Belances::where('type_id', $money)
            ->where('account_id', $accounts_id)
            ->first();

        if ($balance) {
            $balance->belance += ($totalAmount - $paidAmount);
            $balance->save();
        } else {
            $balance = Belances::create([
                'type_id' => $money,
                'user_id' => $validatedData['arr'][0]['user_id'],
                'belance' => ($totalAmount - $paidAmount),
                'account_id' => $accounts_id,
            ]);
        }

        
        $mainBalance = Belances::where('account_id', 1)
            ->where('type_id', $money)
            ->firstOrFail();

        $moneyRecord = Moneys::where('id', $money)->firstOrFail();


        $mainBalance->belance -= $totalAmount;
        $mainBalance->save();

        $moneyRecord->cach -= $paidAmount;
        $moneyRecord->save();

        $bill = Purchasebills::create([
            'accounts_id' => $balance->id,
            'user_id' => $validatedData['arr'][0]['user_id'],
            'total' => $totalAmount,
            'PaidAmount' => $paidAmount,
            'Remain' => $totalAmount - $paidAmount,
            'dateInsert' => $dateInsert,
            'type' => 'purchase',
            'money_id' => $money,
        ]);

        Report::create([
            'isdelete' => 0,
            'user_id' => $validatedData['arr'][0]['user_id'],
            'discription' => 'By Bill Number of ' . $bill->id,
            'amount' => $totalAmount - $paidAmount,
            'date_created' => $dateInsert,
            'type' => 'deposite',
            'account_id' => $balance->id,
            'bill_id' => $bill->id,
        ]);

        $this->createPurchaseItems($validatedData['arr'], $bill->id, $purchase_money);
        $bill->load('accounts.account','money');

        DB::commit();

        return response()->json([
            'message' => 'Purchase created successfully',
            'bill' => $bill,
            'belance' => new BelanceResource($balance),
            'mainbelance' => new BelanceResource($mainBalance),
            'moneys' => new MoneyResource($moneyRecord)
        ], 201);
    }

    /**
     * Handle new customer purchase
     */
    private function handleNewPurchase($validatedData, $money, $totalAmount, $customerName, $dateInsert, $purchase_money)
    {
        $mainBalance = Belances::where('account_id', 1)
            ->where('type_id', $money)
            ->firstOrFail();

        $moneyRecord = Moneys::where('id', $money)->firstOrFail();

        $mainBalance->belance -= $totalAmount;
        $mainBalance->save();

        $moneyRecord->cach -= $totalAmount;
        $moneyRecord->save();

        $bill = Purchasebills::create([
            'temp_customer' => $customerName,
            'user_id' => $validatedData['arr'][0]['user_id'],
            'total' => $totalAmount,
            'PaidAmount' => $totalAmount,
            'Remain' => 0,
            'dateInsert' => $dateInsert,
            'type' => 'purchase',
            'money_id' => $money,
        ]);

        $this->createPurchaseItems($validatedData['arr'], $bill->id, $purchase_money);

        DB::commit();

        $bill->load('accounts', 'money', 'user');

        return response()->json([
            'message' => 'Purchase created successfully',
            'bill' => $bill,
            'mainbelance' => new BelanceResource($mainBalance),
            'moneys' => new MoneyResource($moneyRecord)
        ], 201);
    }

    /**
     * Create purchase items
     */
    private function createPurchaseItems($items, $billId, $money)
    {
        foreach ($items as $data) {
            $this->addItem($data, $money);
            
            $purchaseData = [
                'bill_id' => $billId,
                'remainqty' => $data['qty'],
                'money_id' => $money,
                'dateInsert' => $this->parseDate($data['dateInsert']),
                'expiry_date' => isset($data['expiry_date']) ? $this->parseDate($data['expiry_date']) : null,
            ] + $data;

            Purchase::create($purchaseData);
        }
    }

    /**
     * Add item to stock
     */
    protected function addItem($data, $money)
    {
        $product = Item::where('id', $data['stocks_id'])->firstOrFail();
        $product->qty += (int) $data['qty'];
        $product->purchase_price=(int) $data['purchase_price'];
        
        $type = Sidebar::first();
        if ($type && $type->type === 'gold') {
            $product->weight = bcadd($product->weight, $data['weight'] ?? 0, 3);
        }
        
        $product->save();

        $cabinStock = CabinStock::where('item_id', $data['stocks_id'])
            ->where('cobin_id', $data['cobin_id'])
            ->where('purchase_price', $data['purchase_price'])
            ->where('money_id', $money)
            ->first();

        if ($cabinStock) {
            $cabinStock->qty += (int) $data['qty'];
            if ($type && $type->type === 'gold') {
                $cabinStock->weight = bcadd($cabinStock->weight, $data['weight'] ?? 0, 3);
            }
            $cabinStock->save();
        } else {
            CabinStock::create([
                'item_id' => $data['stocks_id'],
                'cobin_id' => $data['cobin_id'],
                'qty' => $data['qty'],
                'weight' => $data['weight'] ?? 0,
                'dateInsert' => $this->parseDate($data['dateInsert']),
                'rate' => $product->rate,
                'user_id' => $data['user_id'],
                'purchase_price' => $data['purchase_price'],
                'expiry_date' => isset($data['expiry_date']) ? $this->parseDate($data['expiry_date']) : null,
                'description' => $data['description'] ?? null,
                'money_id' => $money
            ]);
        }

        return $product;
    }

    /**
     * Remove item from stock
     */
    protected function deleteItem($data, $money)
    {
        $product = Item::where('id', $data['stocks_id'])->firstOrFail();
        $product->qty -= (int) $data['qty'];
        
        $type = Sidebar::first();
        if ($type && $type->type === 'gold') {
            $product->weight = bcsub($product->weight, $data['weight'] ?? 0, 3);
        }
        
        $product->save();

        $cabinStock = CabinStock::where('item_id', $data['stocks_id'])
            ->where('cobin_id', $data['cobin_id'])
            ->where('purchase_price', $data['purchase_price'])
            ->where('money_id', $money)
            ->first();

        if ($cabinStock) {
            $cabinStock->qty -= (int) $data['qty'];
            if ($type && $type->type === 'gold') {
                $cabinStock->weight = bcsub($cabinStock->weight, $data['weight'] ?? 0, 3);
            }
            
            if ($cabinStock->qty <= 0) {
                $cabinStock->delete();
            } else {
                $cabinStock->save();
            }
        }
    }

    /**
     * Display the specified purchase.
     */
    public function show($billId)
    {
        try {
            $purchases = Purchase::with(['bill.money', 'user', 'bill.accounts.account', 'stock'])
                ->whereHas('bill', function ($query) use ($billId) {
                    $query->where('id', $billId);
                })
                ->orderBy('dateInsert', 'desc')
                ->get();

            if ($purchases->isEmpty()) {
                return response()->json(['message' => 'No data found for this bill.'], 404);
            }

            $firstPurchase = $purchases->first();
            $bill = $firstPurchase->bill;

            $groupedData = [
                'bill' => $bill,
                'money' => $bill->money,
                'purchase' => $purchases->toArray(),
            ];

            return response()->json($groupedData);

        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to fetch purchase: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified purchase.
     */
    public function update($id, Request $request)
    {
        DB::beginTransaction();
        
        try {
            $type = $request->query('type');
            
            if ($type === 'delete') {
                return $this->handlePurchaseDelete($request,$id);
            } else {
                return $this->handlePurchaseUpdate($request);
            }

        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'error' => 'Failed to update purchase: ' . $e
            ], 500);
        }
    }

    /**
     * Handle purchase deletion
     */
    private function handlePurchaseDelete(Request $request,$id)
    {

        $data = $request->validate([
            'money' => '',
            'bill_id' => '',
        ]);

        $purchases = Purchase::where('bill_id', $id)->get();
        $bill = Purchasebills::where('id', $id)->firstOrFail();
        $bill->isdelete = 1;
        $bill->save();
        foreach ($purchases as $purchase) {
            $this->deleteItem($purchase, $bill->money_id);
        }


        $money = Moneys::where('id', $bill->money_id)->firstOrFail();
        $money->cach += $bill->PaidAmount;
        $money->save();

        $balance = Belances::where('id', $bill->accounts_id)->first();
        if ($balance) {
            $balance->belance -= ($bill->total - $bill->PaidAmount);
            $balance->save();
        }

        $mainBalance = Belances::where('type_id', $money->id)
            ->where('account_id', 1)
            ->firstOrFail();
        $mainBalance->belance += $bill->total;
        $mainBalance->save();

        $report = Report::where('bill_id', $bill->id)->first();
        if ($report) {
            $report->isdelete = 1;
            $report->save();
        }

        DB::commit();

        return response()->json([
            'message' => 'Purchase successfully deleted'
        ], 200);
    }

    /**
     * Handle purchase update
     */
    private function handlePurchaseUpdate(Request $request)
    {
        // This is a complex method that would need similar refactoring
        // For brevity, I'm showing the structure - you would apply similar patterns as above
        
        // Extract all query parameters
        $prevMoneyId = $request->query('prevMoney');
        $billId = $request->query('Bill_id');
        $existing = $request->query('Exesting');
        $money = $request->query('money');
        $accounts_id = $request->query('accounts_id');
        $paidAmount = $request->query('PaidAmount', 0);
        $totalAmount = $request->query('TotalAmount', 0);
        $customerName = $request->query('CustomerName');
        $primaryPaidAmount = $request->query('primaryPaidAmount', 0);
        $accountsId = $request->query('Accounts_id');
        $primaryTotalAmount = $request->query('primaryTotalAmount', 0);
        $dateInsert = $request->query('DateInsert');



        $validatedData = $request->validate([
                'arr' => 'required|array',
                'arr.*.stocks_id' => 'required',
                'arr.*.accounts_id' => 'nullable',
                'arr.*.qty' => 'required|numeric|min:0',
                'arr.*.weight' => 'nullable',
                'arr.*.dateInsert' => 'required',
                'arr.*.rate' => 'nullable|numeric|min:0',
                'arr.*.user_id' => 'required',
                'arr.*.isdelete' => 'nullable|boolean',
                'arr.*.purchase_price' => 'required|numeric|min:0',
                'arr.*.sell_price' => 'nullable|numeric|min:0',
                'arr.*.expiry_date' => 'nullable',
                'arr.*.description' => 'nullable',
                'arr.*.tepm_customer' => 'nullable',
                'arr.*.money_id' => 'nullable',
                'arr.*.cobin_id' => 'nullable',
                'arr.*.remainqty' => 'nullable',
            ]);

        // Handle previous balance adjustments
        $this->adjustPreviousBalance($accountsId, $prevMoneyId, $primaryTotalAmount, $primaryPaidAmount, $billId);

        // Delete existing purchases and adjust stock
        $purchasesToDelete = Purchase::where('bill_id', $billId)->get();
        foreach ($purchasesToDelete as $purchase) {
            $this->deleteItem($purchase, $prevMoneyId);
            $purchase->delete();
        }

        if ($existing === 'ok') {
            return $this->handleExistingPurchaseUpdate($validatedData, $money, $accounts_id, $paidAmount, $totalAmount, $dateInsert, $billId);
        } else {
            return $this->handleNewPurchaseUpdate($validatedData, $money, $paidAmount, $totalAmount, $customerName, $dateInsert, $billId);
        }
    }




    // 


    /**
 * Handle existing purchase update
 */
private function handleExistingPurchaseUpdate($validatedData, $money, $accounts_id, $paidAmount, $totalAmount, $dateInsert, $billId)
{
    $balance=[];
    $balance = Belances::where('type_id', $money)
        ->where('account_id', $accounts_id)
        ->first();

    $bill = Purchasebills::where('id', $billId)->firstOrFail();
    $mainBalance = Belances::where('account_id', 1)
        ->where('type_id', $money)
        ->firstOrFail();
    $moneyRecord = Moneys::where('id', $money)->firstOrFail();

    if ($balance) {
        // Update existing balance
        // $balance->belance -= ($totalAmount - $paidAmount);  
        $balance->belance += (int)($totalAmount - $paidAmount);
        $balance->save();
        
        $bill->accounts_id = $balance->id;
    } 
    
    else {
        // Create new balance
        $balance = Belances::create([
            'type_id' => $money,
            'user_id' => $validatedData['arr'][0]['user_id'],
            'belance' => ($totalAmount - $paidAmount),
            'account_id' => $accounts_id,
        ]);
        
        $bill->accounts_id = $balance->id;
    }

    // Update main balance
    $mainBalance->belance -= $totalAmount;
    $mainBalance->save();

    // Update money record
    $moneyRecord->cach -= $paidAmount;
    $moneyRecord->save();

    // Update bill
    $bill->user_id = $validatedData['arr'][0]['user_id'];
    $bill->total = $totalAmount;
    $bill->PaidAmount = $paidAmount;
    $bill->Remain = $totalAmount - $paidAmount;
    $bill->dateInsert = $this->parseDate($dateInsert);
    $bill->money_id = $money;
    $bill->temp_customer = null;
    $bill->save();

    // Update or create report
    $report = Report::where('bill_id', $billId)->first();
    if ($report) {
        $report->amount = $totalAmount - $paidAmount;
        $report->account_id = $balance->id;
        $report->save();
    } else {
        Report::create([
            'isdelete' => 0,
            'user_id' => $validatedData['arr'][0]['user_id'],
            'discription' => 'By Bill Number of ' . $billId,
            'amount' => $totalAmount - $paidAmount,
            'date_created' => $this->parseDate($dateInsert),
            'type' => 'deposite',
            'account_id' => $balance->id,
            'bill_id' => $billId,
        ]);
    }

    // Create new purchase items
    $this->createPurchaseItems($validatedData['arr'], $billId, $money);
    $bill->load('accounts.account','money');
    DB::commit();

    return response()->json([
        'message' => 'Purchase updated successfully',
        'bill' => $bill,
        'belance' => new BelanceResource($balance),
        'mainbelance' => new BelanceResource($mainBalance),
        'moneys' => new MoneyResource($moneyRecord)
    ], 200);
}

/**
 * Handle new customer purchase update
 */
private function handleNewPurchaseUpdate($validatedData, $money, $paidAmount, $totalAmount, $customerName, $dateInsert, $billId)
{
    if ($money == 0 || $money == null) {
        return response()->json(['error' => 'Invalid money type'], 400);
    }

    $bill = Purchasebills::where('id', $billId)->firstOrFail();

    $mainBalance = Belances::firstOrCreate(
        ['account_id' => 1, 'type_id' => $money],
        ['belance' => 0]
    );

    $moneyRecord = Moneys::where('id', $money)->firstOrFail();

    // Update main balance
    $mainBalance->belance -= $totalAmount;
    $mainBalance->save();

    // Update money record
    $moneyRecord->cach -= $paidAmount;
    $moneyRecord->save();

    // Update bill
    $bill->money_id = $money;
    $bill->total = $totalAmount;
    $bill->PaidAmount = $paidAmount;
    $bill->Remain = 0;
    $bill->dateInsert = $this->parseDate($dateInsert);
    $bill->temp_customer = $customerName;
    $bill->accounts_id = null;
    $bill->save();

    // Delete existing report
    Report::where('bill_id', $billId)->delete();

    // Recreate items
    $this->createPurchaseItems($validatedData['arr'], $billId, $money);

    DB::commit();

    return response()->json([
        'message' => 'Purchase updated successfully',
        'bill' => $bill,
        'mainbelance' => new BelanceResource($mainBalance),
        'moneys' => new MoneyResource($moneyRecord),
    ], 200);
}


    /**
     * Adjust previous balance during update
     */
    private function adjustPreviousBalance($accountsId, $prevMoneyId, $primaryTotalAmount, $primaryPaidAmount, $billId)
    {
        if ($accountsId) {
            $prevBalance = Belances::where('id', $accountsId)->firstOrFail();
            $prevMainBalance = Belances::where('account_id', 1)
                ->where('type_id', $prevBalance->type_id)
                ->firstOrFail();
                
            $prevMainBalance->belance += $primaryTotalAmount;
            $prevMainBalance->save();
            
            $prevBalance->belance -=  (int)($primaryTotalAmount - $primaryPaidAmount);
            $prevBalance->save();
            
            $prevMoney = Moneys::where('id', $prevMoneyId)->firstOrFail();
            $prevMoney->cach += $primaryPaidAmount;
            $prevMoney->save();
            
            Report::where('bill_id', $billId)->delete();
        }else{
            $prevMainBalance = Belances::where('account_id', 1)
                ->where('type_id', $prevMoneyId)
                ->firstOrFail();
                
            $prevMainBalance->belance += $primaryTotalAmount;
            $prevMainBalance->save();

            $prevMoney = Moneys::where('id', $prevMoneyId)->firstOrFail();
            $prevMoney->cach += $primaryPaidAmount;
            $prevMoney->save();
            
            
        }
    }

    /**
     * Parse date safely
     */
    private function parseDate($date)
    {
        if (!$date) return null;
        
        try {
            return Carbon::parse($date)->format('Y-m-d H:i:s');
        } catch (Exception $e) {
            throw new Exception('Invalid date format: ' . $date);
        }
    }

    /**
     * Remove the specified purchase from storage.
     */
    public function destroy($id)
    {
        DB::beginTransaction();
        
        try {
            $purchase = Purchase::findOrFail($id);
            $purchase->delete();

            DB::commit();

            return response()->json([
                'message' => 'Purchase successfully deleted.'
            ], 200);
            
        } catch (Exception $exception) {
            DB::rollBack();
            return response()->json([
                'error' => 'Unexpected error occurred while trying to process your request.'
            ], 500);
        }
    }
}