<?php

namespace App\Helpers;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Collection;

class FileUploadHelper
{
    /**
     * آپلود فایل تکی
     *
     * @param UploadedFile $file
     * @param string $folderName
     * @param bool $usePublicPath
     * @return string|null
     */
 public static function upload(UploadedFile $file, string $folderName, ?bool $usePublicPath = null): ?string
{
    try {
        // نام خاص برای پوشه مشخص شده
        if ($folderName === 'company_pic') {
            $fileName = 'logo.' . $file->getClientOriginalExtension();
        } else {
            $fileName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
        }

        // مسیر ذخیره‌سازی بر اساس محیط
        $usePublicPath = $usePublicPath ?? env('UPLOAD_USE_PUBLIC_PATH', true);

        $destination = $usePublicPath
            ? public_path($folderName)
            : $_SERVER['DOCUMENT_ROOT'] . '/' . $folderName;

        if (!file_exists($destination)) {
            mkdir($destination, 0755, true);
        }

        $file->move($destination, $fileName);

        return asset($folderName . '/' . $fileName);
    } catch (\Exception $e) {
        return null;
    }
}

    /**
     * آپلود چندتایی فایل‌ها
     *
     * @param array|Collection $files
     * @param string $folderName
     * @param bool $usePublicPath
     * @return array
     */
    public static function uploadMultiple($files, string $folderName, bool $usePublicPath = true): array
    {
        $uploadedPaths = [];

        foreach ($files as $file) {
            if ($file instanceof UploadedFile) {
                $path = self::upload($file, $folderName, $usePublicPath);
                if ($path) {
                    $uploadedPaths[] = $path;
                }
            }
        }

        return $uploadedPaths;
    }

    /**
     * حذف فایل با URL
     *
     * @param string $fileUrl
     * @return bool
     */
    public static function delete(string $fileUrl): bool
    {
        $filePath = $_SERVER['DOCUMENT_ROOT'] . parse_url($fileUrl, PHP_URL_PATH);

        if (file_exists($filePath)) {
            return unlink($filePath);
        }

        return false;
    }
}
